/*
 * Copyright European Commission's
 * Taxation and Customs Union Directorate-General (DG TAXUD).
 */
package eu.europa.ec.taxud.cesop.writers;

import javax.xml.stream.XMLStreamException;

import java.io.OutputStream;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;

import eu.europa.ec.taxud.cesop.domain.MessageTypeEnum;
import eu.europa.ec.taxud.cesop.domain.ValidationError;
import eu.europa.ec.taxud.cesop.domain.ValidationErrorTypeEnum;
import eu.europa.ec.taxud.cesop.domain.ValidationResultTypeEnum;
import eu.europa.ec.taxud.cesop.domain.XmlPaymentDataMsg;
import eu.europa.ec.taxud.cesop.domain.XmlTypeAndValue;
import eu.europa.ec.taxud.cesop.utils.ValidationConstants.XML;
import eu.europa.ec.taxud.cesop.xsd.XsdSchema;

import static eu.europa.ec.taxud.cesop.domain.ValidationResultTypeEnum.FULLY_REJECTED;
import static eu.europa.ec.taxud.cesop.domain.ValidationResultTypeEnum.PARTIALLY_REJECTED;
import static eu.europa.ec.taxud.cesop.domain.ValidationResultTypeEnum.VALIDATED;
import static eu.europa.ec.taxud.cesop.utils.LangUtils.isNotBlank;
import static eu.europa.ec.taxud.cesop.utils.ValidationErrorUtils.XML_DATE_TIME_FORMATTER;

/**
 * XML writer for validation message.
 */
public class ValidationMessageXmlWriter extends CesopXmlWriter<OutputStream> {

    /**
     * Constructor.
     *
     * @param outputStream the output stream
     * @throws XMLStreamException in case of exception
     */
    public ValidationMessageXmlWriter(final OutputStream outputStream) throws XMLStreamException {
        super(outputStream);
    }

    public void createValidationMessage(final XmlPaymentDataMsg paymentDataMsg,
            final List<ValidationError> validationErrors) throws XMLStreamException {
        this.writeStartValidationMessage(paymentDataMsg);
        if (paymentDataMsg != null) {
            this.writeMessageSpec(paymentDataMsg, MessageTypeEnum.VLD);
            this.writeEOL();
        }
        this.writeValidationResult(validationErrors, paymentDataMsg == null ? XsdSchema.CURRENT_VERSION : paymentDataMsg.getCesopVersion());
        this.writeEndValidationMessage();
    }

    public void createPingValidationMessage(final XmlPaymentDataMsg paymentDataMsg) throws XMLStreamException {
        this.writeStartValidationMessage(paymentDataMsg);
        if (paymentDataMsg != null) {
            this.writeMessageSpec(paymentDataMsg, MessageTypeEnum.PNG);
        }
        this.writeEndValidationMessage();
    }

    private void writeEndValidationMessage() throws XMLStreamException {
        this.writeEOL();
        this.writeEndElement();
        this.writeEOL();
        this.flush();
    }

    private void writeStartValidationMessage(XmlPaymentDataMsg paymentDataMsg) throws XMLStreamException {
        this.writeStartDocument();
        this.writeEOL();
        this.writeStartElement(XML.CESOP_QNAME);
        this.writeAttribute(XML.ATTRIBUTE_NAME_VERSION, getVersionAttribute(paymentDataMsg));
        this.writeNamespace();
        this.writeEOL();
    }

    private String getVersionAttribute(XmlPaymentDataMsg paymentDataMsg) {
        return paymentDataMsg == null || paymentDataMsg.getCesopVersion() == null ? "" : paymentDataMsg.getCesopVersion();
    }

    private void writeMessageSpec(final XmlPaymentDataMsg paymentDataMsg, final MessageTypeEnum messageType)
            throws XMLStreamException {
        this.writeStartElement(XML.MESSAGE_SPEC_QNAME);
        this.writeEOL();
        this.writeTag(XML.TRANSMITTING_COUNTRY_QNAME, paymentDataMsg.getCountry());
        this.writeTag(XML.MESSAGE_TYPE_QNAME, messageType.getLabel());
        this.writeTag(XML.MESSAGE_TYPE_INDIC_QNAME, paymentDataMsg.getMessageType());
        this.writeTag(XML.MESSAGE_REF_ID_QNAME, Optional.ofNullable(paymentDataMsg.getVldMessageRefId())
                .orElse(UUID.randomUUID().toString()).toUpperCase());
        this.writeTag(XML.CORR_MESSAGE_REF_ID_QNAME, paymentDataMsg.getMessageRefId());

        if (paymentDataMsg.getSendingPspIdentifier() != null) {
            this.writeStartElement(XML.SENDING_PSP_QNAME);
            this.writeEOL();

            Map<String, String> attributes = new HashMap<>();
            attributes.put(XML.ATTRIBUTE_NAME_PSP_ID_TYPE, paymentDataMsg.getSendingPspIdentifierType());
            if (paymentDataMsg.getSendingPspIdentifierOther() != null) {
                attributes.put(XML.ATTRIBUTE_NAME_PSP_ID_OTHER, paymentDataMsg.getSendingPspIdentifierOther());
            }
            this.writeTagWithAttributes(XML.PSP_ID_QNAME, paymentDataMsg.getSendingPspIdentifier(), attributes);

            for (final XmlTypeAndValue pspName : paymentDataMsg.getSendingPspNames()) {
                attributes = new HashMap<>();
                attributes.put(XML.ATTRIBUTE_NAME_NAME_TYPE, pspName.getType());
                if (pspName.getOther() != null) {
                    attributes.put(XML.ATTRIBUTE_NAME_NAME_OTHER, pspName.getOther());
                }
                this.writeTagWithAttributes(XML.NAME_QNAME, pspName.getValue(), attributes);
            }
            this.writeEndElement();
            this.writeEOL();
        }

        this.writeStartElement(XML.REPORTING_PERIOD_QNAME);
        this.writeEOL();
        this.writeTag(XML.QUARTER_QNAME, paymentDataMsg.getReportingPeriod() % 10);
        this.writeTag(XML.YEAR_QNAME, paymentDataMsg.getReportingPeriod() / 10);
        this.writeEndElement();
        this.writeEOL();
        this.writeTag(XML.TIMESTAMP_QNAME, XML_DATE_TIME_FORMATTER.format(
                Optional.ofNullable(paymentDataMsg.getProcessedOn())
                        .map(v -> v.atOffset(ZoneOffset.UTC))
                        .orElse(OffsetDateTime.now())));
        this.writeEndElement();
    }

    private void writeValidationResult(final List<ValidationError> validationErrors, String xsdVersion) throws XMLStreamException {
        this.writeStartElement(XML.VALIDATION_RESULT_QNAME);
        this.writeEOL();

        final ValidationResultTypeEnum validationResultTypeEnum = validationErrors.stream().anyMatch(ValidationError::isFullRejection)
                ? FULLY_REJECTED
                : validationErrors.isEmpty() ? VALIDATED : PARTIALLY_REJECTED;

        this.writeTag(XML.VALIDATION_RESULT_QNAME, validationResultTypeEnum.getCode());
        for (final ValidationError validationError : validationErrors) {
            // to comply with the original xsd version, replace not yet known codes with "Custom error".
            String errorCode = ValidationErrorTypeEnum.isSupported(validationError.getErrorType(), xsdVersion)
                    ? validationError.getErrorCode()
                    : ValidationErrorTypeEnum.CM_TR_9999.getCode();
            this.writeStartElement(XML.VALIDATION_ERRORS_QNAME);
            this.writeEOL();
            this.writeTag(XML.ERROR_CODE_QNAME, errorCode);
            this.writeTag(XML.ERROR_COUNTER_QNAME, validationError.getErrorCounter());
            this.writeTag(XML.ERROR_SHORT_DESC_QNAME, validationError.getErrorShortDesc());
            this.writeTag(XML.ERROR_DESCRIPTION_QNAME, validationError.getErrorLongDesc());
            if (isNotBlank(validationError.getTransactionIdentifier())) {
                this.writeTag(XML.TRANSACTION_IDENTIFIER_QNAME, validationError.getTransactionIdentifier());
            }
            if (isNotBlank(validationError.getDocRefId())) {
                this.writeTag(XML.DOC_REF_ID_QNAME, validationError.getDocRefId());
            }
            this.writeEndElement();
            this.writeEOL();
        }
        this.writeEndElement();
    }
}
